<?php
// api/index.php
// Full Backend Control for WhalesKit App

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

require_once __DIR__ . '/config/db.php';

function send_json($data) {
    echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// 🔥 IMPROVED PRICE FETCHER (Binance -> Bybit Fallback)
function fetch_price($symbol) {
    static $cache = [];
    $symbol = strtoupper($symbol);
    
    // Stablecoins
    if (in_array($symbol, ['USDT', 'USD', 'USDC', 'DAI'])) return 1.0;
    
    // Check Cache
    if (isset($cache[$symbol])) return $cache[$symbol];

    $price = 0.0;
    $pair = $symbol . 'USDT';

    // 1. Try BINANCE API (Fastest & Reliable)
    try {
        $url = "https://api.binance.com/api/v3/ticker/price?symbol=" . $pair;
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2); 
        $res = curl_exec($ch);
        curl_close($ch);
        
        if ($res) {
            $data = json_decode($res, true);
            if (isset($data['price'])) {
                $price = (float)$data['price'];
            }
        }
    } catch (Exception $e) { /* ignore */ }

    // 2. Fallback to BYBIT API (If Binance fails)
    if ($price <= 0) {
        try {
            $url = "https://api.bybit.com/v5/market/tickers?category=linear&symbol=" . $pair;
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 2);
            $res = curl_exec($ch);
            curl_close($ch);

            if ($res) {
                $data = json_decode($res, true);
                if (isset($data['result']['list'][0]['lastPrice'])) {
                    $price = (float)$data['result']['list'][0]['lastPrice'];
                }
            }
        } catch (Exception $e) { /* ignore */ }
    }
    
    // Fallback: If still 0, return 1 to avoid division by zero error, but ideally 0 check handles it later
    return $cache[$symbol] = $price;
}

$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents('php://input'), true);

try {
    if (!isset($pdo)) throw new Exception("DB Connection failed");

    switch ($action) {
        // --- LOGIN ---
        case 'login':
            $email = trim($input['email'] ?? '');
            $pass  = $input['password'] ?? '';
            if (!$email || !$pass) send_json(['success' => false, 'message' => 'Required fields missing']);

            $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user && password_verify($pass, $user['password_hash'])) {
                $token = bin2hex(random_bytes(32));
                send_json([
                    'success' => true,
                    'message' => 'Login successful',
                    'data' => [
                        'id' => $user['id'],
                        'name' => $user['name'],
                        'email' => $user['email'],
                        'role' => $user['role'],
                        'token' => $token,
                        'avatar' => $user['avatar'] ?? 'whale'
                    ]
                ]);
            } else {
                send_json(['success' => false, 'message' => 'Invalid credentials']);
            }
            break;

        // --- DASHBOARD DATA (BTC Calculation Fixed) ---
        case 'get_dashboard_data':
            $email = trim($input['email'] ?? $_GET['email'] ?? '');
            if (!$email) send_json(['success' => false, 'message' => 'Email required']);

            // Fetch User
            $stmt = $pdo->prepare("SELECT id, name, role, balance_usd, pnl_base_usd FROM users WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$user) send_json(['success' => false, 'message' => 'User not found']);

            $userId = $user['id'];
            $walletBalance = (float)$user['balance_usd']; 

            // Fetch Assets
            $assetStmt = $pdo->prepare("SELECT * FROM user_assets WHERE user_id = ?");
            $assetStmt->execute([$userId]);
            $assets = $assetStmt->fetchAll(PDO::FETCH_ASSOC);

            $totalAssetValue = 0.0;
            $assetList = [];

            foreach ($assets as $asset) {
                $symbol = $asset['symbol'];
                $amount = (float)$asset['amount'];
                
                $livePrice = fetch_price($symbol);
                // Fallback to stored value if live price fails
                $currentVal = ($livePrice > 0) ? ($amount * $livePrice) : (float)$asset['usd_value'];

                $totalAssetValue += $currentVal;

                $assetList[] = [
                    'symbol' => $symbol,
                    'amount' => $amount,
                    'usd_value' => $currentVal,
                    'pnl_usd' => (float)$asset['pnl_usd'],
                    'pnl_percent' => (float)$asset['pnl_percent']
                ];
            }

            // 🔥 TOTAL CALCULATION
            $grandTotal = $walletBalance + $totalAssetValue;

            // 🔥 BTC EQUIVALENT CALCULATION
            $btcPrice = fetch_price('BTC'); // Get Live BTC Price
            
            // Avoid division by zero
            if ($btcPrice > 0) {
                $btcEq = $grandTotal / $btcPrice;
            } else {
                $btcEq = 0.0;
            }

            send_json([
                'success' => true,
                'data' => [
                    'name' => $user['name'],
                    'role' => $user['role'],
                    'balance_usd' => $grandTotal,
                    'btc_equivalent' => $btcEq, // This will now be correct
                    'btc_price_debug' => $btcPrice, // For checking if price is fetched
                    'pnl_usd' => (float)$user['pnl_base_usd'],
                    'total_usd' => $grandTotal,
                    'funding_usd' => $walletBalance,
                    'unified_usd' => 0.00,
                    'assets' => $assetList
                ]
            ]);
            break;

        // --- ADMIN: GET USERS ---
        case 'get_users':
            $stmt = $pdo->query("SELECT id, name, email, role, vip_level, balance_usd, created_at FROM users ORDER BY id DESC");
            send_json(['success' => true, 'data' => $stmt->fetchAll(PDO::FETCH_ASSOC)]);
            break;

        // --- ADMIN: ADD USER ---
        case 'add_user':
            $name = trim($input['name'] ?? '');
            $email = trim($input['email'] ?? '');
            $pass = $input['password'] ?? '';
            $role = $input['role'] ?? 'user';

            $chk = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $chk->execute([$email]);
            if ($chk->fetch()) send_json(['success' => false, 'message' => 'Email exists']);

            $stmt = $pdo->prepare("INSERT INTO users (name, email, password_hash, role, balance_usd, created_at) VALUES (?, ?, ?, ?, 0, NOW())");
            if ($stmt->execute([$name, $email, password_hash($pass, PASSWORD_DEFAULT), $role])) {
                send_json(['success' => true, 'message' => 'User created']);
            }
            break;

        // --- ADMIN: DELETE USER ---
        case 'delete_user':
            $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
            if ($stmt->execute([$input['id'] ?? 0])) send_json(['success' => true, 'message' => 'Deleted']);
            break;

        // --- ADMIN: FULL UPDATE ---
        case 'admin_update_full':
            $id = $input['id'] ?? 0;
            $fields = []; $params = [];
            
            if (isset($input['name'])) { $fields[] = "name=?"; $params[] = $input['name']; }
            if (isset($input['email'])) { $fields[] = "email=?"; $params[] = $input['email']; }
            if (isset($input['role'])) { $fields[] = "role=?"; $params[] = $input['role']; }
            if (isset($input['vip_level'])) { $fields[] = "vip_level=?"; $params[] = $input['vip_level']; }
            if (isset($input['balance_usd'])) { $fields[] = "balance_usd=?"; $params[] = $input['balance_usd']; }
            if (!empty($input['new_password'])) {
                $fields[] = "password_hash=?";
                $params[] = password_hash($input['new_password'], PASSWORD_DEFAULT);
            }
            
            $params[] = $id;
            $sql = "UPDATE users SET " . implode(', ', $fields) . " WHERE id = ?";
            if ($pdo->prepare($sql)->execute($params)) send_json(['success' => true, 'message' => 'Updated']);
            break;

        default:
            send_json(['success' => false, 'message' => 'Invalid action']);
    }

} catch (Exception $e) {
    send_json(['success' => false, 'message' => 'Error', 'error' => $e->getMessage()]);
}
?>