<?php
// api/lib/bybit_client.php
// Bybit V5 REST client (HMAC SHA256, SIGN-TYPE=2)
// Ref: Bybit official api-usage-examples (V5)

function bybit_base_url(): string {
    // Use production. If you need testnet, change to https://api-testnet.bybit.com
    $u = getenv('BYBIT_BASE_URL');
    return $u ? rtrim($u, '/') : 'https://api.bybit.com';
}

function bybit_gen_signature(string $timestamp, string $apiKey, string $recvWindow, string $payload, string $apiSecret): string {
    $paramStr = $timestamp . $apiKey . $recvWindow . $payload;
    return hash_hmac('sha256', $paramStr, $apiSecret);
}

/**
 * @return array{ok:bool,http:int|null,body_raw:string|null,body_json:array|null,error:string|null}
 */
function bybit_http_request(
    string $method,
    string $endpoint,
    array $queryOrBody,
    bool $auth,
    ?string $apiKey = null,
    ?string $apiSecret = null,
    int $recvWindowMs = 5000
): array {
    $method = strtoupper($method);
    $base = bybit_base_url();

    $timestamp = (string)round(microtime(true) * 1000);
    $recvWindow = (string)$recvWindowMs;

    $url = $base . $endpoint;

    $payload = '';
    $body = null;

    if ($method === 'GET') {
        // signature payload must be querystring (no leading '?')
        $payload = http_build_query($queryOrBody);
        if ($payload !== '') {
            $url .= '?' . $payload;
        }
    } else {
        $body = json_encode($queryOrBody, JSON_UNESCAPED_SLASHES);
        $payload = $body ?: '';
    }

    $headers = [
        'Content-Type: application/json',
    ];

    if ($auth) {
        if (!$apiKey || !$apiSecret) {
            return [
                'ok' => false,
                'http' => null,
                'body_raw' => null,
                'body_json' => null,
                'error' => 'Missing API credentials',
            ];
        }

        $sign = bybit_gen_signature($timestamp, $apiKey, $recvWindow, $payload, $apiSecret);

        $headers[] = 'X-BAPI-API-KEY: ' . $apiKey;
        $headers[] = 'X-BAPI-SIGN: ' . $sign;
        $headers[] = 'X-BAPI-SIGN-TYPE: 2';
        $headers[] = 'X-BAPI-TIMESTAMP: ' . $timestamp;
        $headers[] = 'X-BAPI-RECV-WINDOW: ' . $recvWindow;
    }

    if (!function_exists('curl_init')) {
        return [
            'ok' => false,
            'http' => null,
            'body_raw' => null,
            'body_json' => null,
            'error' => 'cURL not available on server',
        ];
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 12);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 8);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    if ($method !== 'GET') {
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body ?? '');
    }

    $raw = curl_exec($ch);
    $err = curl_error($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($raw === false) {
        return [
            'ok' => false,
            'http' => $http ?: null,
            'body_raw' => null,
            'body_json' => null,
            'error' => $err ?: 'Unknown cURL error',
        ];
    }

    $json = json_decode($raw, true);

    return [
        'ok' => $http >= 200 && $http < 300,
        'http' => $http,
        'body_raw' => $raw,
        'body_json' => is_array($json) ? $json : null,
        'error' => ($http >= 200 && $http < 300) ? null : ('HTTP ' . $http),
    ];
}

function bybit_public_get(string $endpoint, array $query = []): array {
    return bybit_http_request('GET', $endpoint, $query, false);
}

function bybit_private_get(string $endpoint, array $query, string $apiKey, string $apiSecret): array {
    return bybit_http_request('GET', $endpoint, $query, true, $apiKey, $apiSecret);
}

function bybit_private_post(string $endpoint, array $body, string $apiKey, string $apiSecret): array {
    return bybit_http_request('POST', $endpoint, $body, true, $apiKey, $apiSecret);
}
