<?php
// api/lib/crypto.php
// ---------------------------------------------------------
// Encryption helper for storing sensitive values in DB.
// ---------------------------------------------------------
// ✅ সেট করুন (recommended) সার্ভার ENV এ:
//    APP_CRYPT_KEY = a-long-random-secret-at-least-32-chars
//
// NOTE: This is not a replacement for proper secret management.
// It is a practical improvement over plain-text storage.

function app_crypt_key(): string {
    $k = getenv('APP_CRYPT_KEY');
    if ($k && strlen($k) >= 32) return $k;

    // Fallback (⚠️ CHANGE THIS!)
    // Keep it 32+ characters.
    return 'CHANGE_ME__SET_APP_CRYPT_KEY_ENV__PLEASE__32CHARS_MIN';
}

function encrypt_text(string $plain): string {
    if ($plain === '') return '';

    $key = hash('sha256', app_crypt_key(), true); // 32 bytes
    $iv  = random_bytes(12); // recommended for GCM
    $tag = '';

    $cipher = openssl_encrypt(
        $plain,
        'aes-256-gcm',
        $key,
        OPENSSL_RAW_DATA,
        $iv,
        $tag
    );

    if ($cipher === false) {
        // fallback to CBC if GCM isn't available
        $iv2 = random_bytes(16);
        $cipher2 = openssl_encrypt(
            $plain,
            'aes-256-cbc',
            $key,
            OPENSSL_RAW_DATA,
            $iv2
        );
        if ($cipher2 === false) return '';

        $payload = [
            'v' => 1,
            'alg' => 'aes-256-cbc',
            'iv' => base64_encode($iv2),
            'ct' => base64_encode($cipher2),
        ];
        return base64_encode(json_encode($payload));
    }

    $payload = [
        'v' => 1,
        'alg' => 'aes-256-gcm',
        'iv' => base64_encode($iv),
        'tag' => base64_encode($tag),
        'ct' => base64_encode($cipher),
    ];

    return base64_encode(json_encode($payload));
}

function decrypt_text(string $encoded): string {
    if ($encoded === '') return '';

    $raw = base64_decode($encoded, true);
    if ($raw === false) return '';

    $payload = json_decode($raw, true);
    if (!is_array($payload) || !isset($payload['alg'], $payload['iv'], $payload['ct'])) return '';

    $key = hash('sha256', app_crypt_key(), true);

    $alg = (string)$payload['alg'];
    $iv  = base64_decode((string)$payload['iv'], true);
    $ct  = base64_decode((string)$payload['ct'], true);

    if ($iv === false || $ct === false) return '';

    if ($alg === 'aes-256-gcm') {
        $tag = isset($payload['tag']) ? base64_decode((string)$payload['tag'], true) : false;
        if ($tag === false) return '';
        $plain = openssl_decrypt(
            $ct,
            'aes-256-gcm',
            $key,
            OPENSSL_RAW_DATA,
            $iv,
            $tag
        );
        return $plain === false ? '' : $plain;
    }

    if ($alg === 'aes-256-cbc') {
        $plain = openssl_decrypt(
            $ct,
            'aes-256-cbc',
            $key,
            OPENSSL_RAW_DATA,
            $iv
        );
        return $plain === false ? '' : $plain;
    }

    return '';
}
