<?php
// api/wallet/create_withdrawal.php
ini_set('display_errors', 0);
error_reporting(E_ALL);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../db.php';

try {
    $pdo = getDb();
} catch (Throwable $e) {
    echo json_encode(['success' => false, 'message' => 'DB connection failed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

$email       = isset($input['email']) ? trim($input['email']) : '';
$symbol      = 'USDT';
$address     = isset($input['address']) ? trim($input['address']) : '';
$network     = 'TRC20';
$accountType = isset($input['account_type']) ? trim($input['account_type']) : 'funding';
$amount      = isset($input['amount']) ? (float)$input['amount'] : 0.0;

if ($email === '' || $address === '' || $amount <= 0) {
    echo json_encode(['success' => false, 'message' => 'Missing fields']);
    exit;
}
if (!in_array($accountType, ['funding','unified'], true)) {
    echo json_encode(['success' => false, 'message' => 'Invalid account type']);
    exit;
}

$fee       = 1.5;
$minAmount = 2.6;

if ($amount < $minAmount) {
    echo json_encode([
        'success' => false,
        'message' => 'Minimum withdrawal amount is ' . $minAmount . ' USDT',
    ]);
    exit;
}

try {
    // ensure tables exist
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS wallet_balances (
          id INT AUTO_INCREMENT PRIMARY KEY,
          user_id INT NOT NULL,
          account_type ENUM('funding','unified') NOT NULL,
          symbol VARCHAR(16) NOT NULL,
          amount DECIMAL(32,8) NOT NULL DEFAULT 0,
          UNIQUE KEY unique_balance (user_id, account_type, symbol)
        )
    ");
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS withdrawals (
          id INT AUTO_INCREMENT PRIMARY KEY,
          user_id INT NOT NULL,
          symbol VARCHAR(16) NOT NULL,
          amount DECIMAL(32,8) NOT NULL,
          address VARCHAR(255) NOT NULL,
          network VARCHAR(32) NOT NULL,
          fee DECIMAL(32,8) NOT NULL,
          account_type ENUM('funding','unified') NOT NULL,
          status ENUM('pending','approved','rejected') NOT NULL DEFAULT 'pending',
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");

    $pdo->beginTransaction();

    $stmt = $pdo->prepare('SELECT id FROM users WHERE email = ? LIMIT 1');
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'User not found']);
        exit;
    }
    $userId = (int)$user['id'];

    $stmt = $pdo->prepare("
        SELECT id, amount FROM wallet_balances
        WHERE user_id = ? AND account_type = ? AND symbol = 'USDT'
        FOR UPDATE
    ");
    $stmt->execute([$userId, $accountType]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    $current = $row ? (float)$row['amount'] : 0.0;

    $totalDebit = $amount + $fee;
    if ($current < $totalDebit) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Insufficient balance']);
        exit;
    }

    $newAmount = $current - $totalDebit;

    if ($row) {
        $stmt = $pdo->prepare("UPDATE wallet_balances SET amount = ? WHERE id = ?");
        $stmt->execute([$newAmount, $row['id']]);
    } else {
        $stmt = $pdo->prepare("
            INSERT INTO wallet_balances (user_id, account_type, symbol, amount)
            VALUES (?, ?, 'USDT', ?)
        ");
        $stmt->execute([$userId, $accountType, $newAmount]);
    }

    $stmt = $pdo->prepare("
        INSERT INTO withdrawals (user_id, symbol, amount, address, network, fee, account_type)
        VALUES (?, 'USDT', ?, ?, 'TRC20', ?, ?)
    ");
    $stmt->execute([$userId, $amount, $address, $fee, $accountType]);

    $pdo->commit();

    echo json_encode([
        'success' => true,
        'message' => 'Withdrawal request created',
        'data'    => [
            'new_balance' => $newAmount,
            'fee'         => $fee,
        ],
    ]);

} catch (Throwable $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    echo json_encode([
        'success' => false,
        'message' => 'Server error in create_withdrawal',
    ]);
}
