<?php
// quantum/api/markets.php
// Live Spot & Futures with robust fallbacks for HelioHost
// - Providers: Binance → Bybit → CoinCap (spot)
// - Caching + stale-cache serve
// - Sample fallback so client never breaks on first boot

/* ---------- CORS ---------- */
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Access-Control-Allow-Methods: GET, OPTIONS");
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(200); exit; }
header('Content-Type: application/json; charset=utf-8');

/* ---------- Config ---------- */
$CACHE_SECONDS = 8;     // fresh window
$STALE_SECONDS = 3600;  // allow serving stale cache for up to 1 hour if providers fail
$STRICT_SSL    = false; // HelioHost CA issues → keep false
$CACHE_DIR     = __DIR__ . '/.cache';

$spotSymbols    = ['BTCUSDT','ETHUSDT','SOLUSDT','BNBUSDT','XRPUSDT','DOGEUSDT','ADAUSDT','TRXUSDT','TONUSDT','SHIBUSDT'];
$futuresSymbols = ['BTCUSDT','ETHUSDT','SOLUSDT','BNBUSDT','XRPUSDT','DOGEUSDT','ADAUSDT','LINKUSDT','AVAXUSDT','NEARUSDT'];

// Allow override via query (?spot=BTCUSDT,ETHUSDT&futures=BTCUSDT&debug=1)
if (!empty($_GET['spot']))    $spotSymbols    = array_map('trim', explode(',', strtoupper($_GET['spot'])));
if (!empty($_GET['futures'])) $futuresSymbols = array_map('trim', explode(',', strtoupper($_GET['futures'])));
$DEBUG = !empty($_GET['debug']);

/* ---------- Providers ---------- */
// Binance
$binanceSpotBatch = 'https://api.binance.com/api/v3/ticker/24hr?symbols=';
$binanceFutBatch  = 'https://fapi.binance.com/fapi/v1/ticker/24hr?symbols=';
$binanceSpotOne   = 'https://api.binance.com/api/v3/ticker/24hr?symbol=';
$binanceFutOne    = 'https://fapi.binance.com/fapi/v1/ticker/24hr?symbol=';

// Bybit
$bybitSpotAll   = 'https://api.bybit.com/v5/market/tickers?category=spot';
$bybitLinearAll = 'https://api.bybit.com/v5/market/tickers?category=linear';

// CoinCap (spot only)
$coincapAssets  = 'https://api.coincap.io/v2/assets?ids=';

// Symbol → CoinCap id map
$COINCAP_IDS = [
  'BTC'=>'bitcoin','ETH'=>'ethereum','SOL'=>'solana','BNB'=>'binance-coin',
  'XRP'=>'ripple','DOGE'=>'dogecoin','ADA'=>'cardano','TRX'=>'tron',
  'TON'=>'toncoin','SHIB'=>'shiba-inu','LINK'=>'chainlink','AVAX'=>'avalanche',
  'NEAR'=>'near'
];

/* ---------- FS / Cache ---------- */
if (!is_dir($CACHE_DIR)) @mkdir($CACHE_DIR, 0777, true);
function cache_path(string $key): string { global $CACHE_DIR; return $CACHE_DIR . "/$key.json"; }
function cache_get_fresh(string $key, int $ttl) {
  $p = cache_path($key);
  if (is_file($p) && (time() - filemtime($p) < $ttl)) {
    $t = @file_get_contents($p);
    if ($t !== false) { $j = json_decode($t, true); if (is_array($j)) return $j; }
  }
  return null;
}
function cache_get_stale(string $key, int $ttl) {
  $p = cache_path($key);
  if (is_file($p) && (time() - filemtime($p) < $ttl)) {
    $t = @file_get_contents($p);
    if ($t !== false) { $j = json_decode($t, true); if (is_array($j)) return $j; }
  }
  return null;
}
function cache_set(string $key, $val): void { @file_put_contents(cache_path($key), json_encode($val)); }

/* ---------- HTTP helpers ---------- */
function curl_json(string $url, bool $strictSsl, int $timeout = 8): ?array {
  if (!function_exists('curl_init')) return null;
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => $timeout,
    CURLOPT_USERAGENT => 'QuantumMarkets/1.3',
    CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_SSL_VERIFYPEER => $strictSsl,
    CURLOPT_SSL_VERIFYHOST => $strictSsl ? 2 : 0,
  ]);
  $raw = curl_exec($ch);
  curl_close($ch);
  if ($raw === false) return null;
  $j = json_decode($raw, true);
  return is_array($j) ? $j : null;
}
function http_get_json(string $url, bool $strictSsl): ?array {
  $j = curl_json($url, $strictSsl);
  if (is_array($j)) return $j;
  $ctx = stream_context_create(['http' => ['timeout' => 10]]);
  $raw = @file_get_contents($url, false, $ctx);
  if ($raw === false) return null;
  $j = json_decode($raw, true);
  return is_array($j) ? $j : null;
}

/* ---------- Packers ---------- */
function pack_binance_rows(array $rows): array {
  $out = [];
  foreach ($rows as $r) {
    $symbol = strtoupper($r['symbol'] ?? '');
    $last   = (float)($r['lastPrice'] ?? 0);
    if (!$symbol || $last <= 0) continue;
    $out[] = [
      'symbol'     => $symbol,
      'base'       => preg_replace('/USDT$/','',$symbol),
      'quote'      => 'USDT',
      'last'       => $last,
      'change24h'  => (float)($r['priceChangePercent'] ?? 0),
      'volumeUsd'  => (float)($r['quoteVolume'] ?? 0),
    ];
  }
  return $out;
}
function pack_bybit_rows(array $rows): array {
  $out = [];
  foreach ($rows as $r) {
    $symbol = strtoupper($r['symbol'] ?? '');
    $last   = (float)($r['lastPrice'] ?? 0);
    if (!$symbol || $last <= 0) continue;
    $pcnt   = (float)($r['price24hPcnt'] ?? 0) * 100.0; // fraction → %
    $turn   = (float)($r['turnover24h'] ?? $r['quoteVolume24h'] ?? 0);
    $out[] = [
      'symbol'     => $symbol,
      'base'       => preg_replace('/USDT$/','',$symbol),
      'quote'      => 'USDT',
      'last'       => $last,
      'change24h'  => $pcnt,
      'volumeUsd'  => $turn,
    ];
  }
  return $out;
}
function pack_coincap_rows(array $data): array {
  $out = [];
  foreach ($data as $r) {
    $sym   = strtoupper($r['symbol'] ?? '');
    $price = (float)($r['priceUsd'] ?? 0);
    if (!$sym || $price <= 0) continue;
    $out[] = [
      'symbol'     => $sym . 'USDT',
      'base'       => $sym,
      'quote'      => 'USDT',
      'last'       => $price,
      'change24h'  => (float)($r['changePercent24Hr'] ?? 0),
      'volumeUsd'  => (float)($r['volumeUsd24Hr'] ?? 0),
    ];
  }
  return $out;
}

/* ---------- Binance fetchers ---------- */
function fetch_binance_batch(array $symbols, bool $isFut, bool $strictSsl): ?array {
  global $binanceSpotBatch, $binanceFutBatch;
  $u = ($isFut ? $binanceFutBatch : $binanceSpotBatch) . urlencode(json_encode(array_values($symbols)));
  $j = http_get_json($u, $strictSsl);
  return $j ? pack_binance_rows($j) : null;
}
function fetch_binance_per_symbol(array $symbols, bool $isFut, bool $strictSsl): ?array {
  global $binanceSpotOne, $binanceFutOne;
  $out = [];
  foreach ($symbols as $s) {
    $u = ($isFut ? $binanceFutOne : $binanceSpotOne) . urlencode($s);
    $j = http_get_json($u, $strictSsl);
    if ($j) {
      $row = pack_binance_rows([$j]);
      if ($row) $out[] = $row[0];
    }
    usleep(120000);
  }
  return $out ?: null;
}

/* ---------- Bybit fetchers ---------- */
function fetch_bybit_filtered(string $allUrl, array $wanted, bool $strictSsl): ?array {
  $j = http_get_json($allUrl, $strictSsl);
  if (!$j || ($j['retCode'] ?? 1) != 0) return null;
  $list = $j['result']['list'] ?? [];
  if (!is_array($list)) return null;
  $idx = [];
  foreach ($list as $row) {
    if (!isset($row['symbol'])) continue;
    $idx[strtoupper($row['symbol'])] = $row;
  }
  $sel = [];
  foreach ($wanted as $s) {
    $S = strtoupper($s);
    if (isset($idx[$S])) $sel[] = $idx[$S];
  }
  return $sel ? pack_bybit_rows($sel) : null;
}

/* ---------- CoinCap fetcher (spot only) ---------- */
function fetch_coincap(array $symbols, array $mapIds, bool $strictSsl): ?array {
  $ids = [];
  foreach ($symbols as $symUSDT) {
    $base = preg_replace('/USDT$/','', strtoupper($symUSDT));
    if (isset($mapIds[$base])) $ids[] = $mapIds[$base];
  }
  $ids = array_values(array_unique($ids));
  if (!$ids) return null;
  $u = 'https://api.coincap.io/v2/assets?ids=' . implode(',', $ids);
  $j = http_get_json($u, $strictSsl);
  if (!$j || !isset($j['data']) || !is_array($j['data'])) return null;
  return pack_coincap_rows($j['data']);
}

/* ---------- Orchestrate with cache + fallbacks ---------- */
$debugInfo = ['spot'=>[],'futures'=>[]];

function fetch_with_fallbacks(array $symbols, bool $forFutures, bool $strictSsl, array $mapIds, array &$debugInfo) {
  $debugKey = $forFutures ? 'futures' : 'spot';

  // 1) Binance batch
  $rows = fetch_binance_batch($symbols, $forFutures, $strictSsl);
  if ($rows) { $debugInfo[$debugKey][]='binance_batch'; return $rows; }

  // 2) Binance per-symbol
  $rows = fetch_binance_per_symbol($symbols, $forFutures, false);
  if ($rows) { $debugInfo[$debugKey][]='binance_per_symbol'; return $rows; }

  // 3) Bybit (spot / linear)
  if ($forFutures) {
    $rows = fetch_bybit_filtered($GLOBALS['bybitLinearAll'], $symbols, false);
    if ($rows) { $debugInfo[$debugKey][]='bybit_linear'; return $rows; }
  } else {
    $rows = fetch_bybit_filtered($GLOBALS['bybitSpotAll'], $symbols, false);
    if ($rows) { $debugInfo[$debugKey][]='bybit_spot'; return $rows; }
  }

  // 4) CoinCap (spot only)
  if (!$forFutures) {
    $rows = fetch_coincap($symbols, $mapIds, false);
    if ($rows) { $debugInfo[$debugKey][]='coincap_spot'; return $rows; }
  }

  $debugInfo[$debugKey][]='all_failed';
  return null;
}

$spotKey = 'spot_' . md5(implode(',', $spotSymbols));
$futKey  = 'fut_'  . md5(implode(',', $futuresSymbols));

$spot = cache_get_fresh($spotKey, $CACHE_SECONDS);
if (!$spot) {
  $spot = fetch_with_fallbacks($spotSymbols, false, $STRICT_SSL, $COINCAP_IDS, $debugInfo);
  if ($spot) cache_set($spotKey, $spot);
}

$futures = cache_get_fresh($futKey, $CACHE_SECONDS);
if (!$futures) {
  $futures = fetch_with_fallbacks($futuresSymbols, true, $STRICT_SSL, $COINCAP_IDS, $debugInfo);
  if (!$futures && $spot) { // last resort: mirror spot to requested futures symbols
    $debugInfo['futures'][]='fallback_to_spot';
    $byBase = [];
    foreach ($spot as $r) $byBase[strtoupper($r['base'])] = $r;
    $tmp = [];
    foreach ($futuresSymbols as $sym) {
      $base = preg_replace('/USDT$/','', strtoupper($sym));
      if (isset($byBase[$base])) {
        $row = $byBase[$base];
        $row['symbol'] = $base.'USDT';
        $row['quote']  = 'USDT';
        $tmp[] = $row;
      }
    }
    if ($tmp) $futures = $tmp;
  }
  if ($futures) cache_set($futKey, $futures);
}

/* ---------- FINAL RESPONSES ---------- */
// If both fresh failed, try stale cache (do not 502)
if (!$spot && !$futures) {
  $spot    = cache_get_stale($spotKey, $STALE_SECONDS);
  $futures = cache_get_stale($futKey,  $STALE_SECONDS);
  if ($spot || $futures) {
    echo json_encode(['ok'=>true,'ts'=>time(),'stale'=>true,'spot'=>$spot?:[],'futures'=>$futures?:[]]); exit;
  }
}

// Absolute last fallback: small sample so client can render
if (!$spot && !$futures) {
  $sample = [
    ['symbol'=>'BTCUSDT','base'=>'BTC','quote'=>'USDT','last'=>122096.3,'change24h'=>-1.93,'volumeUsd'=>1440000000],
    ['symbol'=>'ETHUSDT','base'=>'ETH','quote'=>'USDT','last'=>4491.44,'change24h'=>-4.01,'volumeUsd'=>2320000000],
    ['symbol'=>'SOLUSDT','base'=>'SOL','quote'=>'USDT','last'=>222.26,'change24h'=>-4.56,'volumeUsd'=>184000000],
  ];
  echo json_encode(['ok'=>true,'ts'=>time(),'sample'=>true,'spot'=>$sample,'futures'=>$sample]); exit;
}

// Normal success
echo json_encode(['ok'=>true,'ts'=>time(),'spot'=>$spot?:[],'futures'=>$futures?:[]], JSON_UNESCAPED_SLASHES);
